/**
  ******************************************************************************
  * @file    main.c
  * @author  Milandr Application Team
  * @version V2.0.3
  * @date    14/06/2023
  * @brief   Main program body.
  ******************************************************************************
  * <br><br>
  *
  * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
  * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2024 Milandr</center></h2>
  */

/* Includes ------------------------------------------------------------------*/
#include "MDR32F9Qx_ssp.h"
#include "MDR32F9Qx_rst_clk.h"
#include "MDR32F9Qx_dma.h"
#include "MDR32F9Qx_port.h"

/** @addtogroup __MDR32Fx_StdPeriph_Examples MDR32Fx StdPeriph Examples
  * @{
  */

/** @addtogroup __MDR1986VE91_EVAL MDR1986VE91 Evaluation Board
  * @{
  */

/** @addtogroup DMA_SPI_RAM DMA_SPI_RAM
  * @{
  */

/* Private define ------------------------------------------------------------*/
#define BufferSize         32

/* Private variables ---------------------------------------------------------*/
DMA_ChannelInitTypeDef DMA_InitStr;
DMA_CtrlDataInitTypeDef DMA_PriCtrlStr;
SSP_InitTypeDef sSSP;
PORT_InitTypeDef PORT_InitStructure;

uint16_t SrcBuf1[BufferSize];
uint16_t SrcBuf2[BufferSize];
uint16_t DstBuf1[BufferSize];
uint16_t DstBuf2[BufferSize];

volatile ErrorStatus TransferStatus1 = ERROR, TransferStatus2 = ERROR;

/* Private function prototypes -----------------------------------------------*/
void Init_RAM(uint16_t *addr, uint32_t size);
uint32_t ps_rand(uint32_t addr);
ErrorStatus Verif_mem (uint32_t BufSize, uint16_t *pBuffer1, uint16_t *pBuffer2);

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
    RST_CLK_DeInit();

    /* Select HSE as CPU_CLK source */
    RST_CLK_HSEconfig(RST_CLK_HSE_ON);
    while(RST_CLK_HSEstatus() == ERROR);
    RST_CLK_CPUclkSelectionC1(RST_CLK_CPU_C1srcHSEdiv1);
    RST_CLK_CPU_PLLuse(DISABLE);
    RST_CLK_CPUclkSelection(RST_CLK_CPUclkCPU_C3);

    SystemCoreClockUpdate();

    /* Enable peripheral clocks */
    RST_CLK_PCLKcmd((RST_CLK_PCLK_RST_CLK | RST_CLK_PCLK_DMA |
                    RST_CLK_PCLK_SSP1 | RST_CLK_PCLK_SSP2 |
                    RST_CLK_PCLK_PORTF | RST_CLK_PCLK_PORTD | RST_CLK_PCLK_PORTE), ENABLE);

    /* Reset all DMA settings */
    DMA_DeInit();

    /* Reset PORTD settings */
    PORT_DeInit(MDR_PORTD);
    /* Reset PORTE settings */
    PORT_DeInit(MDR_PORTE);
    /* Reset PORTF settings */
    PORT_DeInit(MDR_PORTF);

    /* Configure SSP2 pins: FSS, CLK, RXD, TXD */
    PORT_StructInit(&PORT_InitStructure);
    PORT_InitStructure.PORT_Pin   = (PORT_Pin_12 | PORT_Pin_13);
    PORT_InitStructure.PORT_OE    = PORT_OE_IN;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_OVERRID;
    PORT_InitStructure.PORT_MODE  = PORT_MODE_DIGITAL;
    PORT_InitStructure.PORT_SPEED = PORT_SPEED_FAST;
    PORT_Init(MDR_PORTF, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin   = PORT_Pin_2;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_ALTER;
    PORT_Init(MDR_PORTD, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin   = PORT_Pin_15;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_OVERRID;
    PORT_InitStructure.PORT_OE    = PORT_OE_OUT;
    PORT_Init(MDR_PORTF, &PORT_InitStructure);

    /* Configure SSP1 pins: FSS, CLK, RXD, TXD */
    PORT_InitStructure.PORT_Pin   = PORT_Pin_11;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_OVERRID;
    PORT_InitStructure.PORT_OE    = PORT_OE_IN;
    PORT_Init(MDR_PORTD, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin   = (PORT_Pin_10 | PORT_Pin_12);
    PORT_InitStructure.PORT_OE    = PORT_OE_OUT;
    PORT_Init(MDR_PORTD, &PORT_InitStructure);

    /* Configure PORTE pin 13*/
    PORT_InitStructure.PORT_Pin   = PORT_Pin_13;
    PORT_InitStructure.PORT_OE    = PORT_OE_OUT;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_ALTER;
    PORT_Init(MDR_PORTE, &PORT_InitStructure);

    /* Init RAM */
    Init_RAM(DstBuf1, BufferSize);
    Init_RAM(SrcBuf1, BufferSize);
    Init_RAM(DstBuf2, BufferSize);
    Init_RAM(SrcBuf2, BufferSize);

    /* Reset all SSP settings */
    SSP_DeInit(MDR_SSP1);
    SSP_DeInit(MDR_SSP2);

    SSP_BRGInit(MDR_SSP1,SSP_HCLKdiv16);
    SSP_BRGInit(MDR_SSP2,SSP_HCLKdiv16);

    /* SSP1 MASTER configuration */
    SSP_StructInit (&sSSP);

    sSSP.SSP_SCR        = 0x10;
    sSSP.SSP_CPSDVSR    = 2;
    sSSP.SSP_Mode       = SSP_ModeMaster;
    sSSP.SSP_WordLength = SSP_WordLength16b;
    sSSP.SSP_SPH        = SSP_SPH_1Edge;
    sSSP.SSP_SPO        = SSP_SPO_Low;
    sSSP.SSP_FRF        = SSP_FRF_SPI_Motorola;
    sSSP.SSP_HardwareFlowControl = SSP_HardwareFlowControl_SSE;
    SSP_Init(MDR_SSP1,&sSSP);

    /* SSP2 SLAVE configuration */
    sSSP.SSP_SPH     = SSP_SPH_1Edge;
    sSSP.SSP_SPO     = SSP_SPO_Low;
    sSSP.SSP_CPSDVSR = 12;
    sSSP.SSP_Mode    = SSP_ModeSlave;
    SSP_Init (MDR_SSP2,&sSSP);

    /* Enable SSP1 DMA Rx and Tx request */
    SSP_DMACmd(MDR_SSP1, (SSP_DMA_RXE | SSP_DMA_TXE), ENABLE);
    /* Enable SSP2 DMA Rx and Tx request */
    SSP_DMACmd(MDR_SSP2, (SSP_DMA_RXE | SSP_DMA_TXE), ENABLE);

    /* DMA_Channel_SSP1_RX configuration */
    /* Set Primary Control Data */
    DMA_PriCtrlStr.DMA_SourceBaseAddr = (uint32_t)(&(MDR_SSP1->DR));
    DMA_PriCtrlStr.DMA_DestBaseAddr   = (uint32_t)DstBuf1;
    DMA_PriCtrlStr.DMA_SourceIncSize  = DMA_SourceIncNo;
    DMA_PriCtrlStr.DMA_DestIncSize    = DMA_DestIncHalfword;
    DMA_PriCtrlStr.DMA_MemoryDataSize = DMA_MemoryDataSize_HalfWord;
    DMA_PriCtrlStr.DMA_Mode           = DMA_Mode_Basic;
    DMA_PriCtrlStr.DMA_CycleSize      = BufferSize;
    DMA_PriCtrlStr.DMA_NumContinuous  = DMA_Transfers_4;
    DMA_PriCtrlStr.DMA_SourceProtCtrl = DMA_SourcePrivileged;
    DMA_PriCtrlStr.DMA_DestProtCtrl   = DMA_DestPrivileged;
    /* Set Channel Structure */
    DMA_StructInit(&DMA_InitStr);
    DMA_InitStr.DMA_PriCtrlData         = &DMA_PriCtrlStr;
    DMA_InitStr.DMA_Priority            = DMA_Priority_High;
    DMA_InitStr.DMA_UseBurst            = DMA_BurstClear;
    DMA_InitStr.DMA_SelectDataStructure = DMA_CTRL_DATA_PRIMARY;
    /* Init DMA channel */
    DMA_Init(DMA_Channel_SSP1_RX, &DMA_InitStr);

    /* DMA_Channel_SSP2_RX configuration */
    /* Set Primary Control Data */
    DMA_PriCtrlStr.DMA_SourceBaseAddr = (uint32_t)(&(MDR_SSP2->DR));
    DMA_PriCtrlStr.DMA_DestBaseAddr   = (uint32_t)DstBuf2;
    /* Init DMA channel */
    DMA_Init(DMA_Channel_SSP2_RX, &DMA_InitStr);

    /* DMA_Channel_SSP1_TX configuration */
    /* Set Primary Control Data */
    DMA_PriCtrlStr.DMA_SourceBaseAddr = (uint32_t)SrcBuf1;
    DMA_PriCtrlStr.DMA_DestBaseAddr   = (uint32_t)(&(MDR_SSP1->DR));
    DMA_PriCtrlStr.DMA_SourceIncSize  = DMA_SourceIncHalfword;
    DMA_PriCtrlStr.DMA_DestIncSize    = DMA_DestIncNo;
    DMA_InitStr.DMA_Priority          = DMA_Priority_Default;
    /* Init DMA channel */
    DMA_Init(DMA_Channel_SSP1_TX, &DMA_InitStr);

    /* DMA_Channel_SSP2_TX configuration */
    /* Set Primary Control Data */
    DMA_PriCtrlStr.DMA_SourceBaseAddr = (uint32_t)SrcBuf2;
    DMA_PriCtrlStr.DMA_DestBaseAddr   = (uint32_t)(&(MDR_SSP2->DR));
    /* Init DMA channel */
    DMA_Init(DMA_Channel_SSP2_TX, &DMA_InitStr);

    /* Enable SSP1 */
    SSP_Cmd(MDR_SSP1, ENABLE);
    /* Enable SSP2 */
    SSP_Cmd(MDR_SSP2, ENABLE);

    /* Transfer complete */
    while((SSP_GetFlagStatus(MDR_SSP1, SSP_FLAG_BSY))) {}
    while((SSP_GetFlagStatus(MDR_SSP2, SSP_FLAG_BSY))) {}

    /* Check the correctness of written data */
    TransferStatus1 = Verif_mem ((BufferSize), SrcBuf1, DstBuf2);
    TransferStatus2 = Verif_mem ((BufferSize), SrcBuf2, DstBuf1);
    /* TransferStatus1, TransferStatus2 = SUCCESS, if the data transmitted and received
     are correct */
    /* TransferStatus1, TransferStatus2 = ERROR, if the data transmitted and received
     are different */

    while(1) {}
}

/**
  * @brief  Initialisation memory pseudo random value
  * @param  addr - memory address
  * @param  size - memory size
  * @retval None
  */
void Init_RAM (uint16_t *addr, uint32_t size)
{
    uint32_t i;

    for(i = 0; i < size; i++)
    {
        *addr = ps_rand((uint32_t)addr + i * 4);
        addr++;
    }
}

/**
  * @brief  Generation random values
  * @param  addr: memory address
  * @retval random value
  */
uint32_t ps_rand(uint32_t addr)
{
    uint32_t hash = 0;
    uint32_t i;
    char *key = (char *)&addr;

    for(i = 0; i < 4; i++)
    {
        hash += key[i];
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    for(i = 0; i < 256; i++)
    {
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    hash += (hash << 3);
    hash ^= (hash >> 11);
    hash += (hash << 15);

    return hash;
}

/**
  * @brief  Compares two buffers.
  * @param  pBuffer1, pBuffer2: buffers to be compared.
  * @param  BufSize: buffer's size
  * @retval SUCCESS: pBuffer1 identical to pBuffer2
  *         ERROR: pBuffer1 differs from pBuffer2
  */
ErrorStatus Verif_mem (uint32_t BufSize, uint16_t *pBuffer1, uint16_t *pBuffer2)
{
    uint32_t i;

    for(i = 0; i < BufSize; i++)
    {
        if (*pBuffer1++ != *pBuffer2++)
        {
            return(ERROR);
        }
    }

    return(SUCCESS);
}

/**
  * @brief  Reports the source file name, the source line number
  *         and expression text (if USE_ASSERT_INFO == 2) where
  *         the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @param  expr:
  * @retval None
  */
#if (USE_ASSERT_INFO == 1)
void assert_failed(uint8_t* file, uint32_t line)
{
    /* User can add his own implementation to report the source file name and line number.
       Ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

    /* Infinite loop */
    while(1) {}
}
#elif (USE_ASSERT_INFO == 2)
void assert_failed(uint8_t* file, uint32_t line, const uint8_t* expr)
{
    /* User can add his own implementation to report the source file name, line number and
       expression text.
       Ex: printf("Wrong parameters value (%s): file %s on line %d\r\n", expr, file, line) */

    /* Infinite loop */
    while(1) {}
}
#endif /* USE_ASSERT_INFO */

/** @} */ /* End of group DMA_SPI_RAM */

/** @} */ /* End of group __MDR1986VE91_EVAL */

/** @} */ /* End of group __MDR32Fx_StdPeriph_Examples */


/******************* (C) COPYRIGHT 2024 Milandr *******************************/

/* END OF FILE main.c */

